#!/bin/bash

# Turn on verbose debugging output
#set -x

# CEF 141:
BRANCH=7390

# This must the latest commit date for the specified CEF branch: it is used to
# checkout a valid depot tool (that might be reverted by gclient as needed by
# the CEF version being built).
# See the discussion here:
# https://bitbucket.org/chromiumembedded/cef/pull-requests/341/corresponding-version-of-depot-tools-must/diff
COMMIT_DATE="2025-10-15 23:59:59 +0000"

# The "master" branch is preferred. In case it won't work for the CEF branch
# you are trying to build, try and use the same branch number as for your CEF
# branch, or another branch number in-between your branch and the current CEF
# release branch...
AUTOMATE_BRANCH=master

BUILD_OPTIONS="--minimal-distrib-only --build-target=cefsimple --no-debug-build --x64-build"

DEPOT_TOOLS_OPT=""
if [ "$AUTOMATE_BRANCH" != "master" ] ; then
	DEPOT_TOOLS_OPT="--no-depot-tools-update"
	export DEPOT_TOOLS_UPDATE=0
fi

# When passed '-r' or '--retry', do not re-download/update everything (allows
# to patch the code locally and restart the build).
retrying="no"
if [ "$1" == "--retry" ] || [ "$1" == "-r" ] ; then
	retrying="yes"
fi

# git now needs a large http postBuffer to successfully fetch CEF/Chromium and
# avoid this:
# "error: RPC failed; HTTP 500 curl 22 The requested URL returned error: 500"
git config --global http.postBuffer 1048576000

CURDIR=`pwd`
mkdir -p $CURDIR/CEF/automate $CURDIR/CEF/chromium_git
cd $CURDIR/CEF

echo "================================================="
echo "Download started at: "`date`
echo "================================================="
start=`date +%s`

if ! [ -d depot_tools ] ; then
	git clone https://chromium.googlesource.com/chromium/tools/depot_tools.git
	pushd depot_tools
	git checkout $(git rev-list --all --max-count=1 --before="$COMMIT_DATE")
	popd
fi

if ! [ -f automate/automate-git.py ] ; then
	pushd automate
	wget https://bitbucket.org/chromiumembedded/cef/raw/$AUTOMATE_BRANCH/tools/automate/automate-git.py
	sed -i -e 's/--jobs 16/--jobs 36/' automate-git.py
	popd
fi

# Avoid any conflict with system-wide preloaded library (e.g. libgtk3-nocsd.so)
export LD_PRELOAD=""

# UTF-8 is required by automate-git.py...
my_lang="${LANG%.*}"
my_lang="${my_lang%@*}"
export LANG=$my_lang.UTF-8
export LANGUAGE=$my_lang.UTF-8
export LC_ALL=$my_lang.UTF-8
unset LC_MESSAGES LC_NAME LC_PAPER LC_TELEPHONE LC_COLLATE LC_MONETARY
unset LC_CTYPE LC_IDENTIFICATION LC_TIME LC_ADDRESS LC_MEASUREMENT LC_NUMERIC

function catch()
{
	echo
}
trap 'catch' ERR

pushd chromium_git
export PATH=$CURDIR/CEF/depot_tools:$PATH
export CEF_ARCHIVE_FORMAT=tar.bz2
export CEF_USE_GN=1
export GN_DEFINES="use_sysroot=true is_official_build=true symbol_level=1 is_cfi=false use_thin_lto=true use_gtk=false use_system_libdrm=false proprietary_codecs=true ffmpeg_branding=Chrome angle_build_tests=false angle_has_histograms=false disable_histogram_support=true"
# Important: make sure CEF will leave the malloc allocator alone and use the
# default one, which will be jemalloc in our case.
if (( $BRANCH >= 5414 )) ; then
	export GN_DEFINES="$GN_DEFINES use_partition_alloc_as_malloc=false use_allocator_shim=false"
else
	export GN_DEFINES="$GN_DEFINES use_allocator=none use_allocator_shim=false"
fi
if (( $BRANCH >= 5304 )) ; then
	export GN_DEFINES="$GN_DEFINES chrome_pgo_phase=0"
fi
if (( $BRANCH >= 7151 )) ; then
	export GN_DEFINES="$GN_DEFINES use_siso=false"
else
	export GN_DEFINES="$GN_DEFINES use_qt=false"
fi
if (( $BRANCH >= 5304 )) ; then
	export GN_DEFINES="$GN_DEFINES chrome_pgo_phase=0"
fi
# First step: download the sources and patch them
if [ "$retrying" == "no" ] ; then
	if [ "$1" == "" ] ; then
		python3 ../automate/automate-git.py --download-dir=$CURDIR/CEF/chromium_git --depot-tools-dir=$CURDIR/CEF/depot_tools --branch=$BRANCH --force-clean --no-build --no-debug-build --no-distrib $DEPOT_TOOLS_OPT
	else
		# When passed anything else than a retry option, attempt to interpret
		# it as a new Chromium release tag (e.g. "91.0.4472.164") and update to
		# it
		python3 ../automate/automate-git.py --download-dir=$CURDIR/CEF/chromium_git --depot-tools-dir=$CURDIR/CEF/depot_tools --branch=$BRANCH --fast-update --no-build --no-debug-build --no-distrib $DEPOT_TOOLS_OPT --force-cef-update --chromium-checkout=refs/tags/$1
	fi
	if (( $? != 0 )) ; then
		exit $?
	fi
	# Patch Chromium to build properly against glibc without SHIM
	file="$CURDIR/CEF/chromium_git/chromium/src/base/process/memory_linux.cc"
	if [ -f "$file" ] ; then
		echo "**** Patching the __libc_malloc and __libc_free calls in $file"
		sed -i -e 's/__libc_malloc/malloc/' "$file"
		sed -i -e 's/__libc_free/free/' "$file"
	fi
	# Patch CEF to remove the automatic creation of a "Downloads" directory
	file=`grep -l "if (DownloadPathIsDangerous(" $CURDIR/CEF/chromium_git/cef/libcef/common/*.cc`
	if [ -f "$file" ] ; then
		echo "**** Patching the DownloadPathIsDangerous() call in $file"
		sed -i -e 's/if (DownloadPathIsDangerous(\*result))/if ((false))/' "$file"
	else
		# Note: the above (in libcef) was removed now (at least in CEF 130) and
		# now resides in chromium.
		file=`grep -l "if (DownloadPathIsDangerous(" $CURDIR/CEF/chromium_git/chromium/src/chrome/browser/download/*.cc`
		if [ -f "$file" ] ; then
			echo "**** Patching the DownloadPathIsDangerous() calls in $file"
			sed -i -e 's/if (DownloadPathIsDangerous(path_))/if ((false))/' "$file"
			sed -i -e 's/if (DownloadPathIsDangerous(current_download_dir))/if ((false))/' "$file"
			# Avoid unused function warning and compilation failure due to
			# treating warnings as error in CEF 140 and newer.
			if (( $BRANCH >= 7339 )) ; then
				sed -i -e 's:// Copyright 2012 The Chromium Authors:#pragma clang diagnostic ignored "-Wunused-function":' "$file"
			fi
		fi
	fi
	# Patch Chromium to disable the PDF viewer printing function, since it
	# causes a crash for our GTK-dependency-free build of CEF.
	file="$CURDIR/CEF/chromium_git/chromium/src/chrome/browser/resources/pdf/controller.ts"
	if [ -f "$file" ] ; then
		echo "**** Patching away the print() call in $file"
		sed -i -e "s/this.postMessage_({type: 'print'})/return/" "$file"
	fi
	# Patch CEF 103-104 (105 seems OK now, this file is not even present any
	# more) to remove bogus ShellDialogLinux initialization.
	if (( $BRANCH >= 5060 )) ; then
		file="$CURDIR/CEF/chromium_git/chromium/src/ui/views/linux_ui/linux_ui.cc"
		if [ -f "$file" ] ; then
			echo "**** Patching ShellDialogLinux::SetInstance(g_linux_ui) call in $file"
			sed -i -e 's/ShellDialogLinux::SetInstance(g_linux_ui)//' "$file"
		fi
	fi
fi
# This is new... depot_tools now (2025-04) fails to boostrap Python3, and fails
# to use the vypthon environment it installed...
if ! [ -f "$CURDIR/CEF/depot_tools/python3_bin_reldir.txt" ] ; then
	echo "WARNING: missing depot_tools/python3_bin_reldir.txt file. Fixing..."
	$CURDIR/CEF/depot_tools/ensure_bootstrap &>/dev/null
fi
now=`date +%s`
let delay=$now-$start
echo "================================================="
echo "Download finished at: "`date`
echo "Download time: $delay seconds"
echo "================================================="
# Second step: build for good.
echo "================================================="
echo "Compilation started at: "`date`
echo "================================================="
compile_start=`date +%s`
python3 ../automate/automate-git.py --download-dir=$CURDIR/CEF/chromium_git --depot-tools-dir=$CURDIR/CEF/depot_tools --branch=$BRANCH $BUILD_OPTIONS --no-update --no-depot-tools-update --force-build --force-distrib
popd
now=`date +%s`
let delay=$now-$compile_start
let delay2=$now-$start
echo "================================================="
echo "Compilation stopped at: "`date`
echo "Compilation time: $delay seconds"
echo "Total build time: $delay2 seconds"
echo "================================================="

# Cleanup the virtual Python cache installed by the CEF build system: 2GB of
# now useless data !
rm -rf ~/.vpython*
rm -rf /var/tmp/$(whoami)/vpython*
