/*
	@brief Dullahan - a headless browser rendering engine
		   based around the Chromium Embedded Framework

	Copyright (c) 2017, Linden Research, Inc.

	Permission is hereby granted, free of charge, to any person obtaining a copy
	of this software and associated documentation files (the "Software"), to deal
	in the Software without restriction, including without limitation the rights
	to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
	copies of the Software, and to permit persons to whom the Software is
	furnished to do so, subject to the following conditions:

	The above copyright notice and this permission notice shall be included in
	all copies or substantial portions of the Software.

	THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
	OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
	THE SOFTWARE.
*/

#include <iostream>

#include "dullahan_impl.h"

#include "SDL2/SDL_keycode.h"

// Some VK_* virtual key codes for Win32, used for the SDL2 to Win32 virtual
// keys translation.
#define VK_UNKNOWN		0x00
#define VK_BACK			0x08
#define VK_TAB			0x09
#define VK_CLEAR		0x0C
#define VK_RETURN		0x0D
#define VK_SHIFT		0x10
#define VK_CONTROL		0x11
#define VK_MENU			0x12
#define VK_PAUSE		0x13
#define VK_CAPITAL		0x14
#define VK_ESCAPE		0x1B
#define VK_PRIOR		0x21
#define VK_NEXT			0x22
#define VK_END			0x23
#define VK_HOME			0x24
#define VK_LEFT			0x25
#define VK_UP			0x26
#define VK_RIGHT		0x27
#define VK_DOWN			0x28
#define VK_INSERT		0x2D
#define VK_DELETE		0x2E
#define VK_HELP			0x2F
#define VK_MULTIPLY		0x6A
#define VK_DIVIDE		0x6F
#define VK_F1			0x70
#define VK_F2			0x71
#define VK_F3			0x72
#define VK_F4			0x73
#define VK_F5			0x74
#define VK_F6			0x75
#define VK_F7			0x76
#define VK_F8			0x77
#define VK_F9			0x78
#define VK_F10			0x79
#define VK_F11			0x7A
#define VK_F12			0x7B
#define VK_F13			0x7C
#define VK_F14			0x7D
#define VK_F15			0x7E
#define VK_OEM_PLUS		0xBB
#define VK_OEM_MINUS	0xBD
#define VK_OEM_PERIOD	0xBE

// Translation table for SDL to Windows key codes
uint32_t SDL1_to_Win[320] = {
	0,   0,   0,   0,   0,   0,   0,   0,   8,   9,   0,   0,   12,  13,  0,   0,
	0,   0,   0,   19,  0,   0,   0,   0,   0,   0,   0,   27,  0,   0,   0,   0,
	32,  0,   0,   0,   0,   0,   0,   222, 0,   0,   0,   0,   188, 189, 190, 191,
	48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  0,   186, 226, 187, 0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   219, 220, 221, 0,   0,
	223, 65,  66,  67,  68,  69,  70,  71,  72,  73,  74,  75,  76,  77,  78,  79,
	80,  81,  82,  83,  84,  85,  86,  87,  88,  89,  90,  0,   0,   0,   0,   46,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	96,  97,  98,  99,  100, 101, 102, 103, 104, 105, 110, 111, 106, 109, 107, 0,
	0,   38,  40,  39,  37,  45,  36,  35,  33,  34,  112, 113, 114, 115, 116, 117,
	118, 119, 120, 121, 122, 123, 124, 125, 126, 0,   0,   0,   144, 20,  145, 161,
	160, 163, 162, 165, 164, 0,   0,   91,  92,  0,   0,   47,  44,  0,   3,   93
};

// Translates SDL modifiers into CEF ones
uint32_t SDLtoCEFmodifiers(uint32_t sdl_modifiers)
{
	uint32_t cef_modifiers = 0;
	cef_modifiers |= (sdl_modifiers & KMOD_CAPS)	? 0x0001 : 0;
	cef_modifiers |= (sdl_modifiers & KMOD_SHIFT)	? 0x0002 : 0;
	cef_modifiers |= (sdl_modifiers & KMOD_CTRL)	? 0x0004 : 0;
#if 0	// not used
	cef_modifiers |= (sdl_modifiers & KMOD_LALT)	? 0x0008 : 0;
	cef_modifiers |= (sdl_modifiers & KMOD_RALT)	? 0x0010 : 0;
#endif
	cef_modifiers |= (sdl_modifiers & KMOD_NUM)		? 0x0100 : 0;
	return cef_modifiers;
}

void dullahan_impl::nativeKeyboardEventLin(dullahan::EKeyEvent key_event,
										   uint32_t native_scan_code,
										   uint32_t native_virtual_key,
										   uint32_t native_modifiers)
{
	if (!mBrowser || !mBrowser->GetHost())
	{
		return;
	}

	CefKeyEvent event;
	event.native_key_code = native_virtual_key;
	event.character = native_virtual_key;
	event.unmodified_character = native_virtual_key;

	event.modifiers = SDLtoCEFmodifiers(native_modifiers);
	event.is_system_key = (native_modifiers & KMOD_LALT) != 0;

	// Key pad (SDL1) scan codes from SDLK_KP0 to SDLK_KP_EQUALS
	if (native_scan_code >= 256 && native_scan_code <= 272)
	{
		event.modifiers |= EVENTFLAG_IS_KEY_PAD;
	}
	if (native_scan_code < sizeof(SDL1_to_Win) / sizeof(uint32_t))
	{
		event.windows_key_code = SDL1_to_Win[native_scan_code];
	}
	else
	{
		event.windows_key_code = 0;
	}

	if (key_event == dullahan::KE_KEY_DOWN)
	{
		event.type = KEYEVENT_RAWKEYDOWN;
		mBrowser->GetHost()->SendKeyEvent(event);
		if (event.character)
		{
			event.type = KEYEVENT_CHAR;
			mBrowser->GetHost()->SendKeyEvent(event);
		}
	}
	else if (key_event == dullahan::KE_KEY_UP)
	{
		event.native_key_code |= 0xC0000000;
		event.type = KEYEVENT_KEYUP;
		mBrowser->GetHost()->SendKeyEvent(event);
	}
}

// SDL2

static std::map<uint32_t, uint32_t> sSDL2toWin;

// Maps SDLK_ virtual keys to Windows VK_ virtual keys.
// Text is handled via unicode input (SDL_TEXTINPUT event) and does not need to
// be translated into VK_ values as those match already.
//static
uint32_t mapSDL2toWin(uint32_t sld2_virtual_key)
{
	if (sSDL2toWin.empty())
	{
		sSDL2toWin[SDLK_UNKNOWN] = (uint32_t)VK_UNKNOWN;
		sSDL2toWin[SDLK_BACKSPACE] = (uint32_t)VK_BACK;
		sSDL2toWin[SDLK_TAB] = (uint32_t)VK_TAB;
		sSDL2toWin[SDLK_CLEAR] = (uint32_t)VK_CLEAR;
		sSDL2toWin[SDLK_RETURN] = (uint32_t)VK_RETURN;
		sSDL2toWin[SDLK_PAUSE] = (uint32_t)VK_PAUSE;
		sSDL2toWin[SDLK_ESCAPE] = (uint32_t)VK_ESCAPE;
		sSDL2toWin[SDLK_DELETE] = (uint32_t)VK_DELETE;
		sSDL2toWin[SDLK_KP_PERIOD] = (uint32_t)VK_OEM_PERIOD;
		sSDL2toWin[SDLK_KP_DIVIDE] = (uint32_t)VK_DIVIDE;
		sSDL2toWin[SDLK_KP_MULTIPLY] = (uint32_t)VK_MULTIPLY;
		sSDL2toWin[SDLK_KP_MINUS] = (uint32_t)VK_OEM_MINUS;
		sSDL2toWin[SDLK_KP_PLUS] = (uint32_t)VK_OEM_PLUS;
		sSDL2toWin[SDLK_KP_ENTER] = (uint32_t)VK_RETURN;
		sSDL2toWin[SDLK_UP] = (uint32_t)VK_UP;
		sSDL2toWin[SDLK_DOWN] = (uint32_t)VK_DOWN;
		sSDL2toWin[SDLK_RIGHT] = (uint32_t)VK_RIGHT;
		sSDL2toWin[SDLK_LEFT] = (uint32_t)VK_LEFT;
		sSDL2toWin[SDLK_INSERT] = (uint32_t)VK_INSERT;
		sSDL2toWin[SDLK_HOME] = (uint32_t)VK_HOME;
		sSDL2toWin[SDLK_END] = (uint32_t)VK_END;
		sSDL2toWin[SDLK_PAGEUP] = (uint32_t)VK_PRIOR;
		sSDL2toWin[SDLK_PAGEDOWN] = (uint32_t)VK_NEXT;
		sSDL2toWin[SDLK_F1] = (uint32_t)VK_F1;
		sSDL2toWin[SDLK_F2] = (uint32_t)VK_F2;
		sSDL2toWin[SDLK_F3] = (uint32_t)VK_F3;
		sSDL2toWin[SDLK_F4] = (uint32_t)VK_F4;
		sSDL2toWin[SDLK_F5] = (uint32_t)VK_F5;
		sSDL2toWin[SDLK_F6] = (uint32_t)VK_F6;
		sSDL2toWin[SDLK_F7] = (uint32_t)VK_F7;
		sSDL2toWin[SDLK_F8] = (uint32_t)VK_F8;
		sSDL2toWin[SDLK_F9] = (uint32_t)VK_F9;
		sSDL2toWin[SDLK_F10] = (uint32_t)VK_F10;
		sSDL2toWin[SDLK_F11] = (uint32_t)VK_F11;
		sSDL2toWin[SDLK_F12] = (uint32_t)VK_F12;
		sSDL2toWin[SDLK_F13] = (uint32_t)VK_F13;
		sSDL2toWin[SDLK_F14] = (uint32_t)VK_F14;
		sSDL2toWin[SDLK_F15] = (uint32_t)VK_F15;
		sSDL2toWin[SDLK_CAPSLOCK] = (uint32_t)VK_CAPITAL;
		sSDL2toWin[SDLK_RSHIFT] = (uint32_t)VK_SHIFT;
		sSDL2toWin[SDLK_LSHIFT] = (uint32_t)VK_SHIFT;
		sSDL2toWin[SDLK_RCTRL] = (uint32_t)VK_CONTROL;
		sSDL2toWin[SDLK_LCTRL] = (uint32_t)VK_CONTROL;
		sSDL2toWin[SDLK_RALT] = (uint32_t)VK_MENU;
		sSDL2toWin[SDLK_LALT] = (uint32_t)VK_MENU;
		sSDL2toWin[SDLK_HELP] = (uint32_t)VK_HELP;
	}

	std::map<uint32_t, uint32_t>::const_iterator it =
		sSDL2toWin.find(sld2_virtual_key);
	return it != sSDL2toWin.end() ? it->second : sld2_virtual_key;
}

void dullahan_impl::nativeKeyboardEventLin2(dullahan::EKeyEvent key_event,
											uint32_t native_virtual_key,
											uint32_t native_modifiers,
											bool keypad_input)
{
	if (!mBrowser || !mBrowser->GetHost())
	{
		return;
	}

	uint32_t key_data = mapSDL2toWin(native_virtual_key);

	CefKeyEvent event;

	event.modifiers = 0;
	if (keypad_input)
	{
		event.modifiers |= EVENTFLAG_IS_KEY_PAD;
	}
	event.is_system_key = (native_modifiers & KMOD_LALT) != 0;

	if (key_event == dullahan::KE_KEY_CHAR)
	{
		event.character = key_data;
		event.type = KEYEVENT_CHAR;
	}
	else
	{
		event.windows_key_code = key_data;

		if (key_event == dullahan::KE_KEY_DOWN)
		{
			event.type = KEYEVENT_RAWKEYDOWN;
		}
		else if (key_event == dullahan::KE_KEY_UP)
		{
			event.native_key_code = 0xC0000000;
			event.type = KEYEVENT_KEYUP;
		}
	}

	mBrowser->GetHost()->SendKeyEvent(event);
}

void dullahan_impl::platformAddCommandLines(CefRefPtr<CefCommandLine> command_line)
{
    if (mForceX11)
    {
        command_line->AppendSwitchWithValue("ozone-platform-hint", "x11");
        command_line->AppendSwitchWithValue("use-angle", "gl");
        command_line->AppendSwitchWithValue("use-cmd-decoder", "passthrough");
    }

    if (getenv("LL_CEF_OPTIONS"))
    {
        std::string options = getenv("LL_CEF_OPTIONS");
        std::string option;
        while (!options.empty())
        {
            size_t i = options.find(' ');
            if (i != std::string::npos)
            {
                option = options.substr(0, i);
                if (++i < options.size())
                {
                    options = options.substr(i);
                }
                else
                {
                    options.clear();
                }
            }
            else
            {
                option = options;
                options.clear();
            }
            if (option.find("--") == 0)
            {
                if (option.size() < 3) continue;
                option = option.substr(2);
            }
            if (option.empty()) continue;
            i = option.find('=');
            if (i != std::string::npos && i + 1 < option.size())
            {
                std::cerr << "dullahan_impl::OnBeforeCommandLineProcessing() adding option: --"
                          << option.substr(0, i) << "=" << option.substr(i + 1)
                          << std::endl;
                command_line->AppendSwitchWithValue(option.substr(0, i),
                                                    option.substr(++i));
            }
            else
            {
                std::cerr << "dullahan_impl::OnBeforeCommandLineProcessing() adding option: --"
                          << option << std::endl;
                command_line->AppendSwitch(option);
            }
        }
    }
}
