#!/usr/bin/env bash

# Turn on verbose debugging output
set -x
# Make errors fatal
set -e

script=`readlink -f "$0" || echo "$0"`
top=`dirname "$script" || echo .`
cd "$top"

source "$top/VERSION"
if [ "$WEBRTC_SDK_VERSION" == "" ] ; then
	echo "Missing version number !"
	exit 1
fi
if [ "$WEBRTC_LL_RELEASE" == "" ] ; then
	echo "Missing release number !"
	exit 1
fi

# Set WEBRTC_SDK_ARCHIVE_* to an empty string to force a rebuild of webrtc-sdk
# from LL's 3p-webrtc-build github repository sources.
#WEBRTC_SDK_ARCHIVE_X86="webrtc-m137.7151.04.9.17354044714-linux64-17354044714.tar.zst"
#WEBRTC_SDK_SITE_X86="https://github.com/secondlife/3p-webrtc-build/releases/download/m137.7151.04.9/"
WEBRTC_SDK_ARCHIVE_X86="webrtc-$WEBRTC_SDK_VERSION.$WEBRTC_LL_RELEASE-linux64.tar.bz2"
WEBRTC_SDK_SITE_X86="http://sldev.free.fr/libraries/sources/"
WEBRTC_SDK_ARCHIVE_ARM="webrtc-$WEBRTC_SDK_VERSION.$WEBRTC_LL_RELEASE-linux-arm64.tar.bz2"
WEBRTC_SDK_SITE_ARM="http://sldev.free.fr/libraries/sources/"

export LC_ALL="C.UTF-8"
export LANG="C.UTF-8"

arch=`uname -m`
if [ "$arch" == "x86_64" ] ; then
	OS="linux64"
	BUILD_ARCH_FLAGS="-m64"
	WEBRTC_SDK_ARCHIVE="$WEBRTC_SDK_ARCHIVE_X86"
	WEBRTC_SDK_SITE="$WEBRTC_SDK_SITE_X86"
elif [ "$arch" == "aarch64" ] ; then
	OS="linux-arm64"
	BUILD_ARCH_FLAGS="-march=armv8-a+fp+simd"
	WEBRTC_SDK_ARCHIVE="$WEBRTC_SDK_ARCHIVE_ARM"
	WEBRTC_SDK_SITE="$WEBRTC_SDK_SITE_ARM"
else
	echo "Unsupported build system architecture, sorry !"
	exit 1
fi

stage="$top/stage"
if [ -d "$stage" ] ; then
	rm -rf "$stage"
fi
mkdir -p $stage

if ! [ -d "include/webrtc" ] ; then
	if [ "$WEBRTC_SDK_ARCHIVE" == "" ] ; then
		WEBRTC_SDK_ARCHIVE=`find . -name "webrtc*-$OS.tar.*" -print`
		if [ "$WEBRTC_SDK_ARCHIVE" == "" ] ; then
			./make-webrtc-sdk.sh
			WEBRTC_SDK_ARCHIVE=`find . -name "webrtc*-$OS.tar.*" -print`
		fi
		tar xf "$WEBRTC_SDK_ARCHIVE"
	else
		if ! [ -f "$WEBRTC_SDK_ARCHIVE" ] ; then
			wget "$WEBRTC_SDK_SITE$WEBRTC_SDK_ARCHIVE"
		fi
		tar xf $WEBRTC_SDK_ARCHIVE
		if [ -f autobuild-package.xml ] ; then
			# This is LL's package...
			rm -f autobuild-package.xml
			mv -f LICENSES/webrtc-license.txt LICENSES/webrtc-sdk.txt
		fi
	fi
fi

if [ -d llwebrtc/build ] ; then
	rm -rf llwebrtc/build
fi
mkdir -p llwebrtc/build
pushd llwebrtc/build &>/dev/null
# Ensure we use a C++20-compliant compiler
gcc_major=`gcc -dumpversion | cut -d '.' -f1`
if (( $gcc_major < 10 )) ; then
	export CXX="clang++"
	export CC="clang"
fi
cmake -G "Unix Makefiles" -DARCH:STRING="$BUILD_ARCH_FLAGS" ..
make
popd &>/dev/null

mkdir -p "$stage/include"
mkdir -p "$stage/lib/release"
cp -a llwebrtc/build/libllwebrtc.* "$stage/lib/release/"
cp -a llwebrtc/llwebrtc.h "$stage/include/"
cp -a LICENSES "$stage/"

pushd "$stage" &>/dev/null
DATE=`date +%Y%m%d`
tarball="llwebrtc-m137-$OS-$DATE.tar.bz2"
tar cjf "../$tarball" *
popd &>/dev/null
md5sum "$tarball"
