#!/usr/bin/env bash

# Turn on verbose debugging output
set -x
# Make errors fatal
set -e

script=`readlink -f "$0" || echo "$0"`
top=`dirname "$script" || echo .`
cd "$top"

source "$top/VERSION"
if [ "$WEBRTC_SDK_VERSION" == "" ] ; then
	echo "Missing version number !"
	exit 1
fi
if [ "$WEBRTC_LL_RELEASE" == "" ] ; then
	echo "Missing release number !"
	exit 1
fi
if [ "$WEBRTC_LL_BRANCH" == "" ] ; then
	echo "Missing branch name !"
	exit 1
fi

# By passing "arm64" in the command line, you can build an arm64 WebRTC SDK on
# a x86_64 system.
if [ "$1" == "arm64" ] ; then
	WEBRTC_CROSS_COMPILE="arm64"
fi

arch=`uname -m`
if [ "$arch" == "x86_64" ] ; then
	if [ "$WEBRTC_CROSS_COMPILE" == "arm64" ] ; then
		echo "Cross-compiling for arm64."
		OS="linux-arm64"
		BUILD_SCRIPT="ubuntu-20.04_armv8"
	else
		OS="linux64"
		BUILD_SCRIPT="ubuntu-20.04_x86_64"
	fi
elif [ "$arch" == "aarch64" ] ; then
	OS="linux-arm64"
	BUILD_SCRIPT="ubuntu-20.04_armv8"
	# Note: we need box64 to run the sysroot environment provided by the WebRTC
	# build system which is x86_64... Below are some box64 parameter you may
	# need to adjust.
	#export BOX64_DYNACACHE=0
	#export BOX64_DYNAREC_BIGBLOCK=0
	#export BOX64_DYNAREC_MALLOC_HACK=2
	#export BOX64_DYNAREC_STRONGMEM=3
else
	echo "Unsupported build system architecture, sorry !"
	exit 1
fi

top=`pwd`

if [ -d 3p-webrtc-build ] ; then
	rm -rf 3p-webrtc-build
fi
git clone https://github.com/secondlife/3p-webrtc-build
pushd 3p-webrtc-build/build &>/dev/null
if [ "$WEBRTC_LL_BRANCH" != "" ] ; then
	git checkout "$WEBRTC_LL_BRANCH"
fi

# Patch run.py and add patch to WebRTC SDK
if [ -f "$top/run-py.patch" ] && ! [ -f .henri-patch-applied.txt ] ; then
	patch -p1 -s <"$top/run-py.patch"
	touch .henri-patch-applied.txt
fi

# Recover WEBRTC_COMMIT
source VERSION
# Build webrtc-sdk
LC_ALL="C.UTF-8" LANG="C.UTF-8" python3 run.py build $BUILD_SCRIPT --commit $WEBRTC_COMMIT

echo "Packaging...."
if [ -d stage ] ; then
	rm -rf stage
fi
stage="`pwd`/stage"
mkdir -p "$stage/include/webrtc" "$stage/lib/release" "$stage/LICENSES"
cp -a LICENSE "$stage/LICENSES/webrtc-sdk.txt"
pushd _source/$BUILD_SCRIPT/webrtc/src
find . -path './third_party' -prune -o -type f \( -name '*.h' \) -print | \
        xargs -I '{}' cp --parents '{}' "$stage/include/webrtc/"
find . -name '*.h' | grep './third_party' | grep -v '/test' | \
        grep -E 'abseil-cpp|boringssl|expat/files|jsoncpp/source/json|libjpeg|libjpeg_turbo|libsrtp|libyuv|libvpx|opus|protobuf|usrsctp/usrsctpout/usrsctpout' | \
        xargs -I '{}' cp --parents '{}' "$stage/include/webrtc/"
find . -name '*.inc' | grep './third_party' | grep -v '/test' | \
        grep -E 'abseil-cpp|boringssl|expat/files|jsoncpp/source/json|libjpeg|libjpeg_turbo|libsrtp|libyuv|libvpx|opus|protobuf|usrsctp/usrsctpout/usrsctpout' | \
        xargs -I '{}' cp --parents '{}' "$stage/include/webrtc/"
find . -name 'version' | grep './third_party' | grep -v '/test' | grep -E 'abseil-cpp' | \
        xargs -I '{}' cp --parents '{}' "$stage/include/webrtc/"
popd &>/dev/null

# Fix -fpermissive/-Wchanges-meaning error when compiling with gcc in WebRTC
# hearders (ADDED to patches and run.py - 2025-09-05):
# sed -i -e 's/virtual const Network\* Network() const = 0;/virtual const webrtc::Network\* Network() const = 0;/' "$stage/include/webrtc/p2p/base/port_interface.h"
# We need an ordered list of symbols, or would fail to get them linked to our
# shared library: this step is missing from run.py (ADDED via patch - 2025-09-02).
#ranlib _build/$BUILD_SCRIPT/release/webrtc/libwebrtc.a
cp -a _build/$BUILD_SCRIPT/release/webrtc/libwebrtc.a "$stage/lib/release/"
pushd "$stage" &>/dev/null
archive="$top/webrtc-${WEBRTC_VERSION}.${WEBRTC_LL_RELEASE}-$OS.tar.bz2"
tar cjf "$archive" .
popd &>/dev/null

popd &>/dev/null

# Cleanup the virtual Python cache installed by the webrtc build system (1.5GB
# of now useless data).
rm -rf ~/.vpython-* ~/.cache/.vpython-* /var/tmp/$(whoami)/vpython*
